﻿// Copyright (c) Rapid Software LLC. All rights reserved.

// Extends chart control.
// Contains classes: TimeSelection, PinchData, ChartPro
// Depends on jquery, hammer, chart.js

// Represents a time selection.
scada.chart.TimeSelection = class {
    // The absolute start coordinate of the selection relative to the document.
    absStartX = 0;
    // The absolute end coordinate of the selection relative to the document.
    absEndX = 0;
    // Indicates whether selection is in progress.
    inProgress = false;
};

// Represents pinch data.
scada.chart.PinchData = class {
    // The zoom scale.
    scale;
    // The X-coordinate of the gesture center relative to the chart.
    centerX;
    // The Y-coordinate of the gesture center relative to the chart.
    centerY;

    constructor(pinchEvent) {
        this.scale = pinchEvent.scale;
        this.centerX = pinchEvent.center.x;
        this.centerY = pinchEvent.center.y;
    };
};

// Represents a professional chart control.
scada.chart.ChartPro = class ChartPro extends scada.chart.Chart {
    // The time selection set by a user.
    _timeSelection = new scada.chart.TimeSelection();
    // The time selection jQuery element.
    _timeSelectionElem = null;
    // The pinch information jQuery element.
    _pinchInfoElem = null;
    // The reset zoom button.
    _resetZoomBtn = null;
    // The localized phrases used by the chart.
    _phrases = {
        ZoomIn: "Zoom In",
        ZoomOut: "Zoom Out",
        ResetZoom: "Reset Zoom"
    };

    constructor(chartElemID) {
        super(chartElemID);
    }

    // Gets a value indicating whether the chart is zoomed.
    get isZoomed() {
        return this._zoomMode;
    }

    // Creates a time selection element if it doesn't exist.
    _initTimeSelectionElem() {
        if (!this._timeSelectionElem) {
            this._timeSelectionElem = $("<div class='chart-time-selection hidden'></div>")
                .css("background-color", this.displayOptions.plotArea.selectionColor);
            this._chartJqElem.append(this._timeSelectionElem);
        }
    }

    // Starts the time selection from the specified point.
    _startTimeSelection(pageX, pageY) {
        this._chartLayout.updateAbsCoordinates(this._canvasJqElem);

        pageX = Math.floor(pageX);
        pageY = Math.floor(pageY);

        if (this._chartLayout.pointInPlotArea(pageX, pageY)) {
            let areaRect = this._chartLayout.absPlotAreaRect;
            let chartOffset = this._chartJqElem.offset();

            this._timeSelection.absStartX = pageX;
            this._timeSelection.absEndX = pageX;
            this._timeSelection.inProgress = true;

            this._initTimeSelectionElem();
            this._timeSelectionElem
                .css({
                    "left": pageX - chartOffset.left,
                    "top": areaRect.top - chartOffset.top,
                    "width": 1,
                    "height": areaRect.height
                })
                .removeClass("hidden");

            return true;
        } else {
            this._cancelTimeSelection();
            return false;
        }
    }

    // Continues the time selection at the specified point.
    _continueTimeSelection(pageX, pageY) {
        pageX = Math.floor(pageX);
        pageY = Math.floor(pageY);

        if (this._timeSelection.inProgress && this._chartLayout.pointInPlotArea(pageX, pageY)) {
            let chartOffset = this._chartJqElem.offset();
            this._timeSelection.absEndX = pageX;
            let startX = this._timeSelection.absStartX;

            if (pageX >= startX) {
                this._timeSelectionElem.css({
                    "left": startX - chartOffset.left,
                    "width": pageX - startX + 1
                });
            } else {
                this._timeSelectionElem.css({
                    "left": pageX - chartOffset.left,
                    "width": startX - pageX + 1
                });
            }
        }
    }

    // Stops the time selection at the specified point and applies the new scale.
    _stopTimeSelection() {
        let timeSel = this._timeSelection;

        if (timeSel.inProgress) {
            this._cancelTimeSelection();

            if (timeSel.absStartX !== timeSel.absEndX) {
                let startX = this._pageXToTrendX(timeSel.absStartX);
                let endX = this._pageXToTrendX(timeSel.absEndX);
                this.setRange(startX, endX);
            }

            this._showHideResetZoomBtn();
        }
    }

    // Cancels the time selection.
    _cancelTimeSelection() {
        this._timeSelection.inProgress = false;
        if (this._timeSelectionElem) {
            this._timeSelectionElem.addClass("hidden");
        }
    }

    // Creates a pinch information element if it doesn't exist.
    _initPinchInfoElem() {
        if (!this._pinchInfoElem) {
            this._pinchInfoElem = $("<div class='chart-trend-hint hidden'></div>");
            this._chartJqElem.append(this._pinchInfoElem);
        }
    }

    // Starts a pinch zooming process.
    _startPinchZoom(pinchData) {
        this._hintEnabled = false;
        this._continuePinchZoom(pinchData);
    }

    // Continues the pinch zooming process.
    _continuePinchZoom(pinchData) {
        this._chartLayout.updateAbsCoordinates(this._canvasJqElem);
        this._initPinchInfoElem();

        // construct zoom info
        let scaleStr = Math.round(pinchData.scale * 10) / 10.0;
        let infoText = (pinchData.scale >= 1 ? this._phrases.ZoomIn : this._phrases.ZoomOut) +
            " " + scaleStr + "x";

        // allow measuring the pinch info element
        let pinchInfoElem = this._pinchInfoElem;
        pinchInfoElem
            .text(infoText)
            .css({
                "left": 0,
                "top": 0,
                "visibility": "hidden"
            })
            .removeClass("hidden");

        // display the pinch info element
        let chartOffset = this._chartJqElem.offset();
        pinchInfoElem.css({
            "left": pinchData.centerX - chartOffset.left - pinchInfoElem.outerWidth() / 2,
            "top": pinchData.centerY - chartOffset.top - scada.chart.ChartLayout.HINT_OFFSET - pinchInfoElem.outerHeight(),
            "visibility": ""
        });
    }

    // Stops the pinch zooming process.
    _stopPinchZoom(pinchData) {
        this._cancelPinchZoom();
        let timeRange = this._getTimeRange(pinchData);
        this.setRange(timeRange.startTime, timeRange.endTime);
        this._showHideResetZoomBtn();
    }

    // Cancels the pinch zooming process.
    _cancelPinchZoom() {
        this._hintEnabled = true;
        this._pinchInfoElem.addClass("hidden");
    }

    // Gets the time range according to the specified pinch data.
    _getTimeRange(pinchData) {
        let centerTime = this._pageXToTrendX(pinchData.centerX);
        let rangeWidth = (this._xAxisTag.max - this._xAxisTag.min) / pinchData.scale;
        let startTime = centerTime - rangeWidth / 2;
        let endTime = startTime + rangeWidth;

        if (startTime < this.timeRange.startTime) {
            let shift = this.timeRange.startTime - startTime;
            startTime = this.timeRange.startTime;
            endTime += shift;
        }

        if (endTime > this.timeRange.endTime) {
            endTime = this.timeRange.endTime;
        }

        let timeRange = new scada.chart.TimeRange();
        timeRange.startTime = startTime;
        timeRange.endTime = endTime;
        return timeRange;
    }

    // Creates a reset zoom button if it doesn't exist.
    _initResetZoomBtn() {
        if (!this._resetZoomBtn) {
            this._resetZoomBtn = $("<div class='chart-reset-zoom hidden' title='" + this._phrases.ResetZoom +
                "'><i class='fa-solid fa-magnifying-glass-minus'></i></div>")
                .css("color", this.displayOptions.plotArea.frameColor);
            this._chartJqElem.append(this._resetZoomBtn);

            const thisObj = this;
            this._resetZoomBtn.on("mousedown", function (event) {
                event.stopPropagation();
                $(this).addClass("hidden");
                thisObj.resetRange();
            });
        }
    }

    // Determines visibility of the reset zoom button.
    _showHideResetZoomBtn() {
        this._initResetZoomBtn();

        if (this._zoomMode) {
            this._resetZoomBtn
                .css({
                    "right": this._chartLayout.width - this._chartLayout.plotAreaRect.right,
                    "top": this._chartLayout.plotAreaRect.top
                })
                .removeClass("hidden");
        } else {
            this._resetZoomBtn.addClass("hidden");
        }
    }

    // Merges the existing chart phrases with the specified phrases.
    mergePhrases(phrases) {
        this._phrases = $.extend(this._phrases, phrases);
    }

    // Binds events to allow scaling.
    bindScalingEvents() {
        if (this._canvasJqElem && this._canvasJqElem.length) {
            const thisObj = this;

            // events for desktop
            $(this._canvasJqElem.parent())
                .off(".scada.chart.scaling")
                .on("touchstart.scada.chart.scaling", function () {
                    // switch off mouse selection for tablets
                    $(this).off("mousedown.scada.chart.scaling");
                })
                .on("mousedown.scada.chart.scaling", function (event) {
                    if (thisObj._startTimeSelection(event.pageX, event.pageY)) {
                        event.preventDefault();
                    }
                })
                .on("mousemove.scada.chart.scaling", function (event) {
                    thisObj._continueTimeSelection(event.pageX, event.pageY);
                })
                .on("mouseup.scada.chart.scaling", function () {
                    thisObj._stopTimeSelection();
                });

            // touch events
            let touchArea = this._canvasJqElem[0];
            let mc = new Hammer.Manager(touchArea, {
                recognizers: [
                    [Hammer.Pinch, { enable: true }]
                ]
            });

            mc
                .on("pinchstart", function (event) {
                    thisObj._startPinchZoom(new scada.chart.PinchData(event));
                })
                .on("pinchmove", function (event) {
                    thisObj._continuePinchZoom(new scada.chart.PinchData(event));
                })
                .on("pinchend", function (event) {
                    thisObj._stopPinchZoom(new scada.chart.PinchData(event));
                })
                .on("pinchcancel", function () {
                    thisObj._cancelPinchZoom();
                });
        }
    }

    // Converts the time point into a date string.
    pointToDateString(timePoint) {
        return this._pointToDateTimeString(timePoint, ChartPro._DATE_OPTIONS);
    }

    // Converts the time point into a time string.
    pointToTimeString(timePoint) {
        return this._pointToDateTimeString(timePoint, ChartPro._TIME_OPTIONS_SEC);
    }
};
