﻿// Contains renderers for basic components.

rs.mimic.BasicButtonRenderer = class extends rs.mimic.RegularComponentRenderer {
    _setBorder(jqObj, border) {
        if (border.width > 0) {
            super._setBorder(jqObj, border);
        } else {
            jqObj.css("border", ""); // default border
        }
    }

    _setCornerRadius(jqObj, cornerRadius) {
        if (cornerRadius.isSet) {
            super._setCornerRadius(jqObj, cornerRadius);
        } else {
            jqObj.css("border-radius", ""); // default radius
        }
    }

    _completeDom(componentElem, component, renderContext) {
        componentElem.append("<div class='basic-button-content'>" +
            "<div class='basic-button-icon'></div>" +
            "<div class='basic-button-text'></div></div>");
    }

    _setClasses(componentElem, component, renderContext) {
        super._setClasses(componentElem, component, renderContext);
        componentElem.addClass("basic-button");
    }

    _setProps(componentElem, component, renderContext) {
        super._setProps(componentElem, component, renderContext);
        let contentElem = componentElem.children(".basic-button-content:first");
        let iconElem = contentElem.children(".basic-button-icon:first");
        let textElem = contentElem.children(".basic-button-text:first");
        let props = component.properties;

        if (props.imageName) {
            this._setBackgroundImage(iconElem, renderContext.getImage(props.imageName));
            this._setSize(iconElem, props.imageSize);
            iconElem.css("display", "");
        } else {
            this._setBackgroundImage(iconElem, null);
            this._setSize(iconElem, { width: 0, height: 0 });
            iconElem.css("display", "none");
        }

        textElem.text(props.text);
    }

    createDom(component, renderContext) {
        let buttonElem = $("<button type='button'></button>")
            .attr("id", "comp" + renderContext.idPrefix + component.id)
            .attr("data-id", component.id);
        this._completeDom(buttonElem, component, renderContext);
        this._setClasses(buttonElem, component, renderContext);
        this._setProps(buttonElem, component, renderContext);
        this._bindEvents(buttonElem, component, renderContext);
        component.dom = buttonElem;
        return buttonElem;
    }
};

rs.mimic.BasicLedRenderer = class extends rs.mimic.RegularComponentRenderer {
    _setBorder(jqObj, border) {
        // do nothing
    }

    _setCornerRadius(jqObj, cornerRadius) {
        // do nothing
    }

    _setLedBorder(componentElem, borderElem, border) {
        super._setBorder(componentElem, null);
        super._setBorder(borderElem, border);
    }

    _setLedCornerRadius(componentElem, borderElem, cornerRadius) {
        super._setCornerRadius(componentElem, cornerRadius);
        super._setCornerRadius(borderElem, cornerRadius);
    }

    _completeDom(componentElem, component, renderContext) {
        $("<div class='basic-led-border'></div>").appendTo(componentElem);
    }

    _setClasses(componentElem, component, renderContext) {
        super._setClasses(componentElem, component, renderContext);
        componentElem.addClass("basic-led");

        if (!component.properties.isSquare) {
            componentElem.addClass("circle");
        }
    }

    _setProps(componentElem, component, renderContext) {
        super._setProps(componentElem, component, renderContext);
        let borderElem = componentElem.children(".basic-led-border:first");
        let props = component.properties;
        this._setLedBorder(componentElem, borderElem, props.border);
        this._setLedCornerRadius(componentElem, borderElem, props.isSquare ? props.cornerRadius : null);
        borderElem.css("opacity", props.borderOpacity / 100);

        // configure area outside rounded corners
        if (renderContext.editMode) {
            componentElem.css("--border-width", 0);
        }
    }
};

rs.mimic.BasicToggleRenderer = class extends rs.mimic.RegularComponentRenderer {
    _completeDom(componentElem, component, renderContext) {
        componentElem.append("<div class='basic-toggle-lever'></div>");
    }

    _setClasses(componentElem, component, renderContext) {
        super._setClasses(componentElem, component, renderContext);
        componentElem.addClass("basic-toggle");

        const BasicTogglePosition = rs.mimic.BasicTogglePosition;
        let position = component.properties.position;
        componentElem.toggleClass("position-not-set", position === BasicTogglePosition.NOT_SET);
        componentElem.toggleClass("position-off", position === BasicTogglePosition.OFF);
        componentElem.toggleClass("position-on", position === BasicTogglePosition.ON);
    }

    _setProps(componentElem, component, renderContext) {
        super._setProps(componentElem, component, renderContext);
        let props = component.properties;
        this._setPadding(componentElem, props.padding);

        let leverElem = componentElem.children(".basic-toggle-lever:first");
        let minSize = Math.min(component.width, component.height);
        let minInnerSize = Math.min(component.innerWidth, component.innerHeight);
        componentElem.css("border-radius", minSize / 2);

        leverElem.css({
            "background-color": props.foreColor,
            "width": minInnerSize,
            "height": minInnerSize
        });
    }

    _setVisualState(componentElem, visualState) {
        super._setVisualState(componentElem, visualState);

        if (visualState.foreColor) {
            componentElem.children(".basic-toggle-lever:first")
                .css("background-color", visualState.foreColor);
        }
    }

    _setOriginalState(componentElem, props) {
        super._setOriginalState(componentElem, props);
        componentElem.children(".basic-toggle-lever:first")
            .css("background-color", props.foreColor);
    }

    setSize(component, width, height) {
        super.setSize(component, width, height);

        // update corner radius
        let componentElem = component.dom;
        let minSize = Math.min(width, height);
        componentElem.css("border-radius", minSize / 2);

        // update lever size
        let leverElem = componentElem.children(".basic-toggle-lever:first");
        let props = component.properties;
        let innerWidth = width - props.border.width * 2 - props.padding.left - props.padding.right;
        let innerHeight = height - props.border.width * 2 - props.padding.top - props.padding.bottom;
        let minInnerSize = Math.min(innerWidth, innerHeight);

        leverElem.css({
            "width": minInnerSize,
            "height": minInnerSize
        });
    }
};

// Registers the renderers. The function name must be unique.
function registerBasicRenderers() {
    let componentRenderers = rs.mimic.RendererSet.componentRenderers;
    componentRenderers.set("BasicButton", new rs.mimic.BasicButtonRenderer());
    componentRenderers.set("BasicLed", new rs.mimic.BasicLedRenderer());
    componentRenderers.set("BasicToggle", new rs.mimic.BasicToggleRenderer());
}

registerBasicRenderers();
